/*****************************************************************************
**  chBinWriter.cpp
**
**      chBinWriter is a chWriter which writes binary chunk files.
**
**	Gigawatt Studios
**	Copyright(C) 2001 - All Rights Reserved
\****************************************************************************/

#include "stdafx.h"
#include "chBinWriter.hpp"

#include "fsFileStream.hpp"

template <class T>
class fsFilePosSaver
{
	public:

		//====================================================================
		//	Create the fsFilePosSaver with by giving it the file whose
		//	position you want saved.
		//====================================================================
		fsFilePosSaver(T& io_File);

		//====================================================================
		//	When the destructor is called, the position of the file when
		//	the constructor was called will be restored.
		//====================================================================
		~fsFilePosSaver();

		//====================================================================
		//	GetPos returns the position saved (counted from the beginning of
		//	the file).
		//====================================================================
		int GetPos() const;

	private:

		T& m_File;
		int m_Pos;
};

template <class T>
inline
fsFilePosSaver<T>::fsFilePosSaver(T& io_File)
:	m_File(io_File),
	m_Pos(io_File.GetFilePos())
{	
}

template <class T>
inline 
fsFilePosSaver<T>::~fsFilePosSaver()
{
	m_File.SetFilePos(m_Pos);
}

template <class T>
inline int 
fsFilePosSaver<T>::GetPos() const
{ 
	return m_Pos;
}


namespace gfFileUtil
{
	//================================================================================
	//================================================================================
	template <class T>
	inline void Read(fsFileStream& i_File, T& i_Val)
	{
		i_File.Read(sizeof(i_Val), &i_Val);
	}

	//================================================================================
	//================================================================================
	template <class T>
	inline void Write(fsFileStream& i_File, T i_Val)
	{
		i_File.Write(sizeof(i_Val), &i_Val);
	}

}


//====================================================================
//====================================================================
chBinWriter::chBinWriter(fsFileStream& io_File)
:	m_File(io_File)
{
}

//====================================================================
//====================================================================
chBinWriter::~chBinWriter()
{
}

//====================================================================
//	WriteChunkHeader writes a chunk header in the file at the
//	current position.
//====================================================================
void chBinWriter::WriteChunkHeader(	chDefs::Name i_Name, 
									chDefs::Version i_Version, 
									bool i_Container)
{
	int header_pos = m_File.GetFilePos();
	m_HeaderLocs.push(header_pos);
	gfFileUtil::Write(m_File, i_Name);
	gfFileUtil::Write(m_File, chDefs::Size(-1));	// don't know the size yet, write a obviously wrong value
	gfFileUtil::Write(m_File, i_Version);
}

//====================================================================
//	FinishChunk will cause the writer to compute the size of the 
//	chunk and finish writing the header.
//====================================================================
void chBinWriter::FinishChunk()
{
//	DBG_ASSERT0( m_HeaderLocs.size() > 0, "FinishChunk w/o corresponding WriteChunkHeader");
	
	fsFilePosSaver<fsFileStream> saver(m_File);
	int header_pos = m_HeaderLocs.top();
	int chunk_size = saver.GetPos() - (header_pos + sizeof(chDefs::Name) + sizeof(chDefs::Size) + sizeof(chDefs::Version));
	
	m_File.SetFilePos(header_pos + sizeof(chDefs::Name));	//skip past the name
//	DBG_ASSERT0(chunk_size >= 0, "Negative chunk size - how did that happen?");

	gfFileUtil::Write(m_File, chDefs::Size(chunk_size));
	m_HeaderLocs.pop();
}

//====================================================================
//	Write functions - These write the data to the chunk file.
//====================================================================
void chBinWriter::Write(envType::Int8 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

void chBinWriter::Write(envType::UInt8 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

void chBinWriter::Write(envType::Int16 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

void chBinWriter::Write(envType::UInt16 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

void chBinWriter::Write(envType::Int32 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

void chBinWriter::Write(envType::UInt32 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

void chBinWriter::Write(envType::Int64 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

void chBinWriter::Write(envType::UInt64 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

void chBinWriter::Write(envType::Float32 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

void chBinWriter::Write(envType::Float64 i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

//====================================================================
//	This Write writes a string called either "TRUE" or "FALSE" based 
//  on a bool value.
//====================================================================
void chBinWriter::Write(bool i_Val)
{
	gfFileUtil::Write(m_File, i_Val);
}

//====================================================================
//	This Write writes a NULL-terminated single-byte character string
//====================================================================
void chBinWriter::Write(const char* i_Val)
{
	int num_to_write = ::strlen(i_Val);
	m_File.Write(num_to_write + 1, i_Val);	// write NULL also
}

//====================================================================
//	This Write writes a single-byte character string with a length
//	given by i_Length.
//====================================================================
void chBinWriter::Write(const char* i_Val, int i_Length)
{
	m_File.Write(i_Length, i_Val);
}

//====================================================================
//	This Write writes a NULL-terminated Unicode character string
//====================================================================
void chBinWriter::Write(const envType::UInt16* i_Val)
{
	int num_to_write = ::wcslen(i_Val);
	m_File.Write(2 * (num_to_write + 1), i_Val);	// write NULL also
}

//====================================================================
//	This Write writes a Unicode character string with a length
//	given by i_Length.
//====================================================================
void chBinWriter::Write(const envType::UInt16* i_Val, int i_Length)
{
	m_File.Write(i_Length * 2, i_Val);
}


//====================================================================
//	Write chunk of binary data; this is only for
//	chBinWriter, it is not a function of chWriter base class
//====================================================================
void chBinWriter::Write(const void* i_Data, int i_NumBytes)
{
	m_File.Write(i_NumBytes, i_Data);
}

//========================================================================
//	GetLocator returns this file's associated locator
//========================================================================
const std::string& chBinWriter::GetLocator() const
{
	return m_File.GetLocator();
}

